-- Return MIDI event stream that configures the device to be used, this will be called _every_ time
-- when the app is launched or the device is connected.
function controller_initialize(applicationName)
	return {{midi={0xB0,0x14,0x7F}},
			{midi={0xB0,0x15,0x7F}},
			{midi={0xB0,0x16,0x7F}},
			{midi={0xB0,0x17,0x7F}},
			{midi={0xB0,0x18,0x7F}},
			{midi={0xB0,0x19,0x3F}},
			{midi={0xB0,0x1A,0x3F}},
			{midi={0xB0,0x1B,0x3F}},
			{midi={0xB0,0x1C,0x3F}},
			{midi={0xB0,0x1D,0x3F}},
			}
end

-- Return MIDI event stream that finializes the device after usage, this will be called _every_ time
-- when the app is terminated.
function controller_finalize()
	return {{midi={0xB0,0x14,0x00}},
			{midi={0xB0,0x15,0x00}},
			{midi={0xB0,0x16,0x00}},
			{midi={0xB0,0x17,0x00}},
			{midi={0xB0,0x18,0x00}},	-- turn all LEDs off
			{midi={0xB0,0x19,0x00}},
			{midi={0xB0,0x19,0x00}},
			{midi={0xB0,0x1A,0x00}},
			{midi={0xB0,0x1B,0x00}},
			{midi={0xB0,0x1C,0x00}},
			{midi={0xB0,0x1D,0x00}},
		}
end

updateObjectsAfterPatchChange = {}

-- Every MIDI event from this device that is an assignment is filtered through this function
function controller_midi_out(midiEvent,name,valueString,color,needsMomentaryFeedback)
	if midiEvent[0] == 0xB0 then
		-- remove already updated LEDs from the table
		updateObjectsAfterPatchChange[midiEvent[1]] = nil
		if midiEvent[1] >= 0x19 and midiEvent[1] <= 0x1D then
			if color ~= nil then
				channelAndController = (midiEvent[1] - 0x19) + 0xB0
				if midiEvent[2] < 64 then
					colorFactor = 8
				else
					colorFactor = 127
				end
				return {midi={0xB0,midiEvent[1],colorFactor,channelAndController,0x0E,color.r * colorFactor, channelAndController,0x0F,color.g * colorFactor, channelAndController,0x10,color.b * colorFactor}}
			end
		end
	end
	return nil	-- filter nothing, we don't change MIDI events
end


-- Return MIDI event stream that configures the device for the next patch to be used
function controller_select_patch(programchangeNumber, patchname, setname, concertname)

	-- turn the LEDs off
	updateObjectsAfterPatchChange[0x14] = {midi={0xB0,0x14,0x00}}
	updateObjectsAfterPatchChange[0x15] = {midi={0xB0,0x15,0x00}}
	updateObjectsAfterPatchChange[0x16] = {midi={0xB0,0x16,0x00}}
	updateObjectsAfterPatchChange[0x17] = {midi={0xB0,0x17,0x00}}
	updateObjectsAfterPatchChange[0x18] = {midi={0xB0,0x18,0x00}}
	updateObjectsAfterPatchChange[0x19] = {midi={0xB0,0x19,0x00}}
	updateObjectsAfterPatchChange[0x1A] = {midi={0xB0,0x1A,0x00}}
	updateObjectsAfterPatchChange[0x1B] = {midi={0xB0,0x1B,0x00}}
	updateObjectsAfterPatchChange[0x1C] = {midi={0xB0,0x1C,0x00}}
	updateObjectsAfterPatchChange[0x1D] = {midi={0xB0,0x1D,0x00}}

	return nil	-- filter nothing, we don't change MIDI events
end

-- Return MIDI event stream that configures the device for the next patch to be used
function controller_select_patch_done(programchangeNumber, patchname, setname, concertname)
	return updateObjectsAfterPatchChange
end

-- Return the list of valid controllers
function controller_names(channel)
	-- Tranzport sends only _one_ controller for the wheel
	if channel == 0 then
		return {
				 [0x0B] = 'Expression Pedal',
		  		 [0x14] = 'Record',           
		  		 [0x15] = 'Play',          
		  		 [0x16] = 'Rewind',         
		  		 [0x17] = 'Forward',        
		  		 [0x18] = 'Return',  
				 [0x19] = 'Effect 1',       
		  		 [0x1A] = 'Effect 2',       
		  		 [0x1B] = 'Effect 3',       
		  		 [0x1C] = 'Effect 4',       
		  		 [0x1D] = 'Effect 5',       
		  		 [0x1E] = 'Next', 
		  		 [0x1F] = 'Previous', 
				}
	else
		return {}
	end
end

-- Return the list of valid controllers
function controller_note_names(channel)
	return {}
end

-- Define the knobs, buttons, etc. of the device
-- This is also used to check if this device is a match (model, manufacturer and/or device inquiry status)
function controller_info()
	return {
		-- model name for this device
		model = 'GiO',
		-- manufacturer name for this device
		manufacturer = 'Apogee Electronics',

		-- Update controllers even if the patch was just updated and not changed, necessary for the displays to update
		always_update = true,

		-- Certain controllers are passed through automatically (Pitch Bend, Modulation, etc)
		auto_passthrough = false,

		-- All buttons, knobs, keyboard, possible pedals are defined here
		items = {
				-- The default midiType is 'Absolute', it only has to be defined, if it is different
				{name='Expression Pedal', label='Expression',	  objectType='Pedal',                        midi={0xB0,0x0B,MIDI_LSB}},
                                                                                                             
				{name='Record',		label='Record',		objectType='Button', midiType='Momentary', midi={0xB0,0x14,MIDI_LSB}, outport='GiO'},
				{name='Return',		label='Return',		objectType='Button', midiType='Momentary', midi={0xB0,0x18,MIDI_LSB}, outport='GiO'},
				{name='Rewind',		label='Rewind',		objectType='Button', midiType='Momentary', midi={0xB0,0x16,MIDI_LSB}, outport='GiO'},
				{name='Forward',	label='Forward',	objectType='Button', midiType='Momentary', midi={0xB0,0x17,MIDI_LSB}, outport='GiO'},
				
				{name='Previous',	label='Previous',	objectType='Button', midiType='Momentary', midi={0xB0,0x1F,MIDI_LSB}},
				{name='Next',		label='Next',		objectType='Button', midiType='Momentary', midi={0xB0,0x1E,MIDI_LSB}},
				
				{name='Effect 1',	label='Effect 1',	objectType='Button', midiType='Momentary', midi={0xB0,0x19,MIDI_LSB}, outport='GiO'},
				{name='Effect 2',	label='Effect 2',	objectType='Button', midiType='Momentary', midi={0xB0,0x1A,MIDI_LSB}, outport='GiO'},
				{name='Effect 3',	label='Effect 3',	objectType='Button', midiType='Momentary', midi={0xB0,0x1B,MIDI_LSB}, outport='GiO'},
				{name='Effect 4',	label='Effect 4',	objectType='Button', midiType='Momentary', midi={0xB0,0x1C,MIDI_LSB}, outport='GiO'},
				{name='Effect 5',	label='Effect 5',	objectType='Button', midiType='Momentary', midi={0xB0,0x1D,MIDI_LSB}, outport='GiO'},

				{name='Play',		label='Play',		objectType='Button', midiType='Momentary', midi={0xB0,0x15,MIDI_LSB}, outport='GiO'},

			}                                                                                                 
		}
end
